define(['angular', 'app', 'lodash'], function (angular, app) {
    "use strict";

    app.service('HealthMoodGraphService', function ($filter, HealthMoodConstants) {
        var service = {};

        var getChartConfig = function(section, title) {
            var chartConfig = {
                options: {
                    chart: {
                        type: 'spline',
                        borderColor: '#EBBA95',
                        borderWidth: 2,
                        borderRadius: 3,
                        zoomType: 'none',
					    pinchType: 'none',
                        height: 450
                    },
                    plotOptions: {
                        series: {
                            marker: { radius: 6 },
                            events: {
                                legendItemClick: function () {
                                    return false;
                                }
                            }
                        }
                    },
                    tooltip : {
                        followTouchMove: false
                    }
                },
                useHighStocks: false,
                title: {text: title},
                credits: { enabled: false },
                loading: false
            };

            return chartConfig;
        };

        var getDateUTC = function(date, truncateTime) {
            return Date.UTC(
                parseInt($filter('date')(date, 'yyyy')),
                parseInt($filter('date')(date, 'MM')) - 1,
                parseInt($filter('date')(date, 'dd')),
                truncateTime ? 0 : parseInt($filter('date')(date, 'HH')),
                truncateTime ? 0 : parseInt($filter('date')(date, 'mm'))
            );
        };

        var initTmpSeries = function(array) {
            var tmp = {};
            angular.forEach(array, function(item) {
                tmp[item] = [];
            });
            return tmp;
        };

        var processGlucoseSeries = function(points) {
            var connectingLineText = 'connecting-line',
                seriesList = angular.copy(HealthMoodConstants.glucose.entry.testConditions);
            
            seriesList.unshift(connectingLineText);
            var tmpSeries = initTmpSeries(seriesList);

            angular.forEach(angular.copy(points).reverse(), function (point) {
                var recordedDate = new Date(point.recorded),
                    recordedDateUTC = getDateUTC(recordedDate),
                    method = point.method,
                    seriesValue = getSeriesValue(point.valueQuantity, null);

                if(tmpSeries[method] && angular.isNumber(seriesValue)) {
                    tmpSeries[method].push({x: recordedDateUTC, y: seriesValue, method: method});
					tmpSeries[connectingLineText].push({x: recordedDateUTC, y: seriesValue, method: method});
                }
            });

            var chartSeries = [];
            for(var method in tmpSeries) {
                chartSeries.push({
                    name: method,
                    data: tmpSeries[method],
                    showInLegend: method !== connectingLineText,
                    type: method === connectingLineText ? 'spline' : 'scatter',
                    marker: {radius: method === connectingLineText ? 1 : 6}
                });
            }

            return chartSeries;
        };        
        
        var processBPPSeries = function(points) {
            var bpText    = 'Blood Pressure', 
                pulseText = 'Pulse',            
                tmpSeries = initTmpSeries([bpText, pulseText]);
            
            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['updated'])),
                    entry = point.entry;

                var systolic = getSeriesValue(entry[0].valueQuantity, null),
                    diastolic = getSeriesValue(entry[1].valueQuantity, null),
                    pulse = getSeriesValue(entry[2].valueQuantity, null);
                
                var ensureDifferential = function(value1, value2) {
                    var lowValue =  (value1 <= value2) ? value1 : value2,
						highValue = (value2 >= value1) ? value2 : value1;
                    return {low: lowValue, high: highValue};
                };
                
                if(systolic && diastolic) {
                    var diff = ensureDifferential(diastolic, systolic);
                    tmpSeries[bpText].push([dateUTC, diff.low, diff.high]);
                }

				if (pulse) {
					tmpSeries[pulseText].push([dateUTC, pulse]);
				}
			});

            return [
                {name: bpText, type: 'columnrange', data: tmpSeries[bpText], color: '#f15c80'},
                {name: pulseText, type: 'spline', data: tmpSeries[pulseText]}
            ];
        };

        var processMoodSeries = function(points) {
            var moodText = 'Mood',
                tmpSeries = initTmpSeries([moodText]);
            
            var getMoodIcon = function(mood) {
                if(mood >= 1 && mood <=  2) return 'url(assets/images/app-icons/mood/mood-verybad-16x16.png)';
                if(mood >= 3 && mood <=  4) return 'url(assets/images/app-icons/mood/mood-bad-16x16.png)';
                if(mood >= 5 && mood <=  6) return 'url(assets/images/app-icons/mood/mood-neutral-16x16.png)';
                if(mood >= 7 && mood <=  8) return 'url(assets/images/app-icons/mood/mood-good-16x16.png)';
                if(mood >= 9 && mood <= 10) return 'url(assets/images/app-icons/mood/mood-verygood-16x16.png)';
            };

            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['recorded'])),
                    seriesValue = getSeriesValue(point.valueQuantity, null);

                if (angular.isNumber(seriesValue)) {
                    tmpSeries[moodText].push({
                        x: dateUTC,
                        y: seriesValue,
                        marker: {symbol: getMoodIcon(seriesValue)}
                    });
                }
            });

            return [
                {name: moodText, type: 'scatter', data: tmpSeries[moodText], color: '#f7a35c'}
            ];
        };
        
        var processCholesterolSeries = function(points) {
            var tmpSeries = initTmpSeries(['Total', 'HDL', 'LDL', 'Triglycerides']);

            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['updated']), true),
                    entry = point.entry;

                var total = getSeriesValue(entry[0].valueQuantity, null),
                    hdl = getSeriesValue(entry[2].valueQuantity, null),
                    ldl = getSeriesValue(entry[1].valueQuantity, null),
                    triglyceride = getSeriesValue(entry[3].valueQuantity, null);

				if (angular.isNumber(total)) {
					tmpSeries['Total'].push([dateUTC, total]);
				}
				if (angular.isNumber(hdl)) {
					tmpSeries['HDL'].push([dateUTC, hdl]);
				}
				if (angular.isNumber(ldl)) {
					tmpSeries['LDL'].push([dateUTC, ldl]);
				}
				if (angular.isNumber(triglyceride)) {
					tmpSeries['Triglycerides'].push([dateUTC, triglyceride]);
				}
            });

            var chartSeries = [];
            for(var name in tmpSeries) {
                chartSeries.push({
                    name: name,
                    data: tmpSeries[name]
                });
            }

            return chartSeries;
        };

        var getSeriesConfig = function(section, points, metaData) {
            if (!points || points.length === 0) { return []; }
            
            if(section === 'glucose') {
                return processGlucoseSeries(points);
            } else if(section === 'bp-pulse') {
                return processBPPSeries(points);            
            } else if(section === 'mood') {
                return processMoodSeries(points);
            } else if(section === 'cholesterol') {
                return processCholesterolSeries(points);
            } else {
                return processSingleSeries(section, points, metaData);
            }
        };

        var processSingleSeries = function(section, points, metaData) {
            var chartSeries = [{
                name: metaData[0].label,
                type: 'spline',
                data: []
            }];

            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['recorded'])),
                    seriesValue = getSeriesValue(point.valueQuantity, null);

                if (angular.isNumber(seriesValue)) {
                    chartSeries[0].data.push([dateUTC, seriesValue]);
                }
            });

            return chartSeries;
        };

		function getSeriesValue(valueQuantity, emptyValue) {
			return (valueQuantity && angular.isDefined(valueQuantity.value) && (angular.isNumber(valueQuantity.value) || valueQuantity.value.length > 0))
				? parseFloat(valueQuantity.value) : emptyValue;
		}
        
        var getXaxisConfig = function (section, filter) {
            var startDate = new Date(filter.startDate),
                endDate = new Date(filter.endDate),
                startDateUTC = Date.UTC(
                    parseInt($filter('date')(startDate, 'yyyy')),
                    parseInt($filter('date')(startDate, 'MM')) - 1,
                    parseInt($filter('date')(startDate, 'dd')),
                    0, 0
                ),
                endDateUTC = Date.UTC(
                    parseInt($filter('date')(endDate, 'yyyy')),
                    parseInt($filter('date')(endDate, 'MM')) - 1,
                    parseInt($filter('date')(endDate, 'dd')) + 1,
                    0, 0
                ),
                title = section === 'cholesterol' ? 'Date' : 'Date/Time';

            return {
                type: 'datetime',
                dateTimeLabelFormats: {
                    millisecond: '%b %e %I:%M %p',
                    second: '%b %e %I:%M %p',
                    minute: '%b %e %I:%M %p',
                    hour: '%b %e %I:%M %p',
                    day: '%b %e',
                    week: '%b %e',
                    month: '%b %Y',
                    year: '%Y'
                },
                minRange: 24 * 3600000,
                minTickInterval: section === 'cholesterol' ? 24 * 3600000 : 3600000,
                tickPixelInterval : 80,
                min: startDateUTC,
                max: endDateUTC,
                title: {
                    text: title, style: {'font-weight': 'bold', 'color': 'black', 'font-size': '12px'}
                }
            };
        };               
        
        var getMoodPlotBands = function() {
            var c1 = 'rgba(68, 170, 213, 0.1)';
            return [
                {from: 1, to: 2,  color: c1},
                {from: 3, to: 4,  color: c1},
                {from: 5, to: 6,  color: c1},
                {from: 7, to: 8,  color: c1},
                {from: 9, to: 10, color: c1}
            ];
        };
        
        var getYaxisConfig = function(section, metaData) {     
            var style = {'font-weight': 'bold', 'color': 'black', 'font-size': '12px'};
            
            if(section === 'bp-pulse') {
                return [
                    {min: 0, tickInterval: 25, title: {text: 'Blood Pressure' + ' (' + metaData[0].units + ')', style: style}},
                    {linkedTo:0,title: {text: 'Pulse' + ' (' + metaData[2].units + ')', style: style}, opposite: true}
                ];                    
            }
            return {
                min: 0, max: section === 'mood' ? 10.5 : null,
				endOnTick: section !== 'mood',
                allowDecimals: false,
                tickInterval: section === 'mood' ? 1 : 25,
                title: {text: metaData[0].label + ' (' + metaData[0].units + ')', style: style},
                plotBands: section === 'mood' ? getMoodPlotBands() : null
            };
        };

        var getTooltipConfig = function (section, metaData) {
            var tooltip = { shared: (section !== 'glucose'), crosshairs: [true] },
                dateFormat = '%A %m/%d/%Y %I:%M %p',
                dateOnlyFormat = '%A %m/%d/%Y';

            if (section === "glucose") {
                tooltip.formatter = function () {
					var currentPoint = this.point || this.points[0].point;
					return Highcharts.dateFormat(dateFormat, this.x) + '<br/>' +
						currentPoint.method + ' ' + metaData[0].label + ': <b>' + this.y + '</b> ' + metaData[0].units;
                };
            } else if (section === "weight") {
                tooltip.formatter = function () {
                    return Highcharts.dateFormat(dateFormat, this.x) + '<br/>' +
                        metaData[0].label + ': <b>' + this.y + '</b> ' + metaData[0].units;
                };
            }  else if (section === 'bp-pulse') {
                tooltip.formatter = function () {
                    var points = this.points,
                        firstString = getTooltipString(points[0]),
                        secondString = getTooltipString(points[1]);

					function getTooltipString(point) {
						if (!angular.isDefined(point)) {
							return '';
						}

						var isBloodPressure = point.point.high,
							unit = isBloodPressure ? metaData[0].units : metaData[2].units,
							value = isBloodPressure ? (point.point.high + '/' + point.point.low) : point.y;
						return point.series.name + ': <b>' + value + '</b> ' + unit + '<br/>';
					}

                    return Highcharts.dateFormat(dateFormat, this.x) + '<br/>' + firstString + secondString;
                };
            } else if (section === "mood") {
                tooltip.formatter = function () {
                    return Highcharts.dateFormat(dateFormat, this.x) + '<br/>' +
                        metaData[0].label + ': <b>' + this.y + '</b> / 10';
                };
            } else if (section === "cholesterol") {
                tooltip.formatter = function () {
					var unit = metaData[0].units,
                        points = this.points,
						string1 = angular.isDefined(points[0]) ? (points[0].series.name + ': <b>' + points[0].y + '</b> ' + unit + '<br/>') : '',
						string2 = angular.isDefined(points[1]) ? (points[1].series.name + ': <b>' + points[1].y + '</b> ' + unit + '<br/>') : '',
						string3 = angular.isDefined(points[2]) ? (points[2].series.name + ': <b>' + points[2].y + '</b> ' + unit + '<br/>') : '',
						string4 = angular.isDefined(points[3]) ? (points[3].series.name + ': <b>' + points[3].y + '</b> ' + unit + '<br/>') : '';

                    return Highcharts.dateFormat(dateOnlyFormat, this.x) + '<br/>' +
                        string1 + string2 + string3 + string4;
                };
            }
            return tooltip;
        };
        
        var getLegendConfig = function(section) {
            return {enabled: section !== 'mood'}
        };

        service.getGraphConfig = function(section, filter, points, metaData, sectionTitle) {
            var title = sectionTitle + " Values from " + filter.startDate + " to " + filter.endDate,
                chartConfig = getChartConfig(section, title);

            chartConfig.options.tooltip = getTooltipConfig(section, metaData);
            chartConfig.series = getSeriesConfig(section, points, metaData);
            chartConfig.xAxis = getXaxisConfig(section, filter);
            chartConfig.yAxis = getYaxisConfig(section, metaData, chartConfig.series);
            chartConfig.options.legend = getLegendConfig(section);

            return chartConfig;
        };

        return service;
    });
});